/**
 * @file nvt.h
 * @author Pavel Brychta, http://www.xpablo.cz
 *
 * Copyright (c) 2014,15,16 Pavel Brychta. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef _NVT_H_
#define _NVT_H_

#include <stdint.h>
#include <stdbool.h>

#define NVT_PARITY_NONE 1
#define NVT_PARITY_EVEN 3
#define NVT_PARITY_ODD 2

#define NVT_STOP_BITS_1 1
#define NVT_STOP_BITS_15 3
#define NVT_STOP_BITS_2 2

#define NVT_INITIAL_SPEED 9600
#define NVT_INITIAL_DATA_BITS 8
#define NVT_INITIAL_PARITY NVT_PARITY_NONE
#define NVT_INITIAL_STOP_BITS NVT_STOP_BITS_1

#define MAX_NVT_CMD_SIZE 64 // maximalni velikost povelu
#define MAX_NVT_RESP_SIZE 64 // maximalni velikost jedne odpovedi na NVT povel
#ifndef NVT_MAX_AYT_LENGTH
 #define NVT_MAX_AYT_LENGTH 64
#endif

typedef void (*putchr_cb)(uint8_t c); // callback pro vystup bytu - smer NVT stream->Seriovy port
typedef void (*netchr_cb)(uint8_t c); // callback pro vystup bytu - smer NVT stream->Network port
typedef void (*setserial_cb)(int32_t speed, int32_t databits, int32_t parity, int32_t stopbits); // nastaveni parametru seriove linky (cislo mensi nez 0 udava, ze parametr je stejny, jako z minuleho volani - kvuli optimalizaci nastavovani)
typedef uint8_t (*setgetcontrol_cb)(uint8_t newcontrol); // nastaveni/ziskani

class nvt
{
  protected:
	enum
	{
	  NVT_IAC_HUNT = 0, // testujeme na IAC
	  NVT_IAC_FOUND, // minuly znak byl IAC
	  NVT_SB_FOUND, // nasli jsme SB - sledujeme povel
	  NVT_IAC_SB_FOUND, // detekovali jsme IAC behem SB
	  NVT_WILL_WONT_DO_DONT_PARAM, // parametr WILL/WONT/DO/DONT
	} _streamstate;

	uint8_t _nvtcmd[MAX_NVT_CMD_SIZE]; // misto pro prikaz, uvozeny <SB> a ukonceny <SE>
	int _cmdindex; // aktualni pozice v nvtcmd
	int _getindex; // zpracovavaci ukazatel
	int wwdd; // posledni detekovany WILL/WONT/DO/DONT

	uint32_t _speed; // aktualni prenosova rychlost
	uint8_t _databits; // aktualni pocet datovych bitu
	uint8_t _parity; // aktualni parita
	uint8_t _stopbits; // pocet stop bitu
	uint8_t _linestatemask; // aktivni maska odesilani zmen stavu
	uint8_t _modemstatemask; // aktivni maska odesilani zmen stavu

	uint8_t _nvttxbuff[MAX_NVT_RESP_SIZE]; // buffer pro sestaveni odpovedi z NVT
	int _txptr; // ukazatel v bufferu pro sestaveni odpovedi

	char _aytstring[NVT_MAX_AYT_LENGTH]; // misto pro AYT retezec

	putchr_cb _putchr; // metoda pro vystup bytu - smer NVT_Stream->Output (seriovy port)
	netchr_cb _netchr; // metoda pro vystup bytu - smer sitovy port
	setserial_cb _setserial; // nastaveni parametru seriove linky
	setgetcontrol_cb _setgetcontrol; // nastaveni/ziskani

	void puttotx(uint8_t c);
	void puttotxesc(uint8_t c);
	void outdata(uint8_t *ptr, size_t size); // vystup proudu dat do _netchr
	int getnvtbyte(uint8_t *dta);
	void donvtcmd(void);
	void terminalbyte(uint8_t dta);
	void subnegotiationbyte(uint8_t dta);
	void terminalescape(uint8_t dta);
	void parsenvtstream(uint8_t dta);
	void handlewwddp(uint8_t p);

  public:
  	void init(setserial_cb setserial, putchr_cb putchr, netchr_cb netchr);
  	void setAYTstring(const char *aytstring);
    void handlestream(uint8_t *ptr, int len);
    void createsendstream(uint8_t *ptr, int len);
    void newguy(void);
    void setSetGetCtrl(setgetcontrol_cb sgc);
};

#endif
