//#include <Arduino.h>

#include <cbuf.h>
#include "nvt.h"
#include "src/uinterval/uinterval.h"

WiFiServer server(VCP_PORT);
WiFiClient client;
cbuf fromNet(3000);
cbuf toNet(3000);
uint8_t sbuf[128]; // vyrovnavaci buffer pro vycitani serioveho portu/sitoveho soketu

nvt netterm; // network virtual terminal objekt

uint8_t nvttx[1500]; // vyrovnavaci vystupni buffer
uInterval nvtt; // casovac pro rizeni odesilani bufferu
uint32_t baudtiming; // vypocitana delka cekani pri paketizaci

int control = 1; // aktivni rezim rizeni vystupu atd... (implicitne je to No Flow Control)
int dtrstate;
int breakstate;
int rtsstate;

int newguy = 0;

// statistika
uint32_t serial_tx; // pocet odeslanych bytu na seriove rozhrani
uint32_t serial_rx; // pocet prijatych bytu ze serioveho rozhrani
uint32_t drop_tx; // pocet vynechanych bytu pri odesilani (preplneny buffer ze site)
uint32_t drop_rx; // pocet vynechanych bytu pri prijmu (preplneny buffer ze serioveho portu)

/* Nastaveni parametru serioveho portu
*/
void setserialport(int32_t speed, int32_t databits, int32_t parity, int32_t stopbits) // nastaveni parametru seriove linky ( cislo mensi nez 0 udava, ze parametr je steny, jako z minuleho volani - kvuli optimalizaci nastavovani)
{
  uint32_t spd;
  int serialMode;
  int totalbits; // celkovy pocet bitu ve slove pro vypocet timeoutu

  if (speed > 0)
    TRACE(TRACE_INFO, F("New serial speed set to %i"), (int)speed);
  spd = abs(speed);

/* Nastaveni poctu datovych bitu
  BITS:
            Value       Data Bit Size
             0           Request Current Data Bit Size
             1           Available for Future Use
             2           Available for Future Use
             3           Available for Future Use
             4           Available for Future Use
             5           5
             6           6
             7           7
             8           8
             9-127       Available for Future Use
*/
  switch (abs(databits))
  {
    case 5:
      serialMode |= UART_NB_BIT_5;
      totalbits = 5 + 1;
      break;

    case 6:
      serialMode |= UART_NB_BIT_6;
      totalbits = 6 + 1;
      break;

    case 7:
      serialMode |= UART_NB_BIT_7;
      totalbits = 7 + 1;
      break;

    default:
      serialMode |= UART_NB_BIT_8;
      totalbits = 8 + 1;
      break;
  }
  if (databits > 0)
    TRACE(TRACE_INFO, F("New databits set to %i"), (int)databits);

/* Nastaveni typu parity
  PARITY:
              1           NONE
              2           ODD
              3           EVEN
              4           MARK
              5           SPACE
*/
  switch (abs(parity))
  {
    case 2:
      serialMode |= UART_PARITY_ODD;
      totalbits += 1;
      if (parity > 0)
        TRACE(TRACE_INFO, F("Parity set to ODD"));
      break;

    case 3:
      serialMode |= UART_PARITY_EVEN;
      totalbits += 1;
      if (parity > 0)
        TRACE(TRACE_INFO, F("Parity set to EVEN"));
      break;

    case 4:
    case 5:
      if (parity > 0)
        TRACE(TRACE_ERROR, F("Unsupported parity type"));
    default:
      if (parity > 0)
        TRACE(TRACE_INFO, F("Parity set to NONE"));
      serialMode |= UART_PARITY_NONE;
      totalbits += 0;
      break;
  }
/* Nastaveni poctu stop bitu
  STOPBITS:
            Value      Stop Bit Size
             0           Request Current Number of Stop Bits
             1           1
             2           2
             3           1.5
             4-127       Available for Future Use
*/
  switch (abs(stopbits))
  {
    case 2:
      serialMode |= UART_NB_STOP_BIT_2;
      totalbits += 2;
      if (stopbits > 0)
        TRACE(TRACE_INFO, F("Stop bits set to 2"));
      break;

    case 3:
      serialMode |= UART_NB_STOP_BIT_15;
      totalbits += 2; // 1,5 je jako dva...
      if (stopbits > 0)
        TRACE(TRACE_INFO, F("Stop bits set to 1.5"));
      break;

    default:
      serialMode |= UART_NB_STOP_BIT_1;
      totalbits += 1;
      if (stopbits > 0)
        TRACE(TRACE_INFO, F("Stop bits set to 1"));
      break;
  }
// podle poctu bitu a prenosove rychlosti spocitame casovou konstantu pro paketovaci casovac
  uint32_t work = (totalbits * 1000000ul * TX_TIMEOUT_MULTIPLIER) / (spd * TX_TIMEOUT_DIVISOR);
  if (baudtiming != work)
    TRACE(TRACE_DEBUG, "Interval set to %i us", (int)work);
  baudtiming = work;
// skutecne zmenime nastaveni serioveho portu
  Serial.flush();
  //  delay(200);
  Serial.end();
  delay(10); // 100
  Serial.begin(spd, (SerialConfig)serialMode);
  Serial.setDebugOutput(false);
  delay(10); // 100
  Serial.flush();
  serial_tx = 0;
  serial_rx = 0;
}

// Odeslani dat na seriovy port
void sputchar(uint8_t c)
{

  if (fromNet.empty() && Serial.availableForWrite())
  {
  	Serial.write(c);
  	++serial_tx;
  }
  else
  {
  	if (!fromNet.full())
  	{
  	  fromNet.write(c);
  	}
  	else
  	  ++drop_tx;
  }
}

// Ulozeni bytu do sitoveho vystupniho bufferu
void nputchar(uint8_t c)
{

  if (!toNet.full())
  {
    toNet.write(c);
    nvtt.set(baudtiming); // timeout pro odeslani bloku dat
  }
  else
    ++drop_rx;
}

/*
       This command is sent by the client to the access server to set
       special com port options. The command can also be sent to query
       the current option value. The value is one octet (byte). The
       value is an index into the following value table:

           Value      Control Commands
             0           Request Com Port Flow Control Setting
                           (outbound/both)
             1           Use No Flow Control (outbound/both)
             2           Use XON/XOFF Flow Control (outbound/both)
             3           Use HARDWARE Flow Control (outbound/both)
             4           Request BREAK State
             5           Set BREAK State ON
             6           Set BREAK State OFF
             7           Request DTR Signal State
             8           Set DTR Signal State ON
             9           Set DTR Signal State OFF
            10           Request RTS Signal State
            11           Set RTS Signal State ON
            12           Set RTS Signal State OFF
            13           Request Com Port Flow Control Setting (inbound)
            14           Use No Flow Control (inbound)
            15           Use XON/XOFF Flow Control (inbound)
            16           Use HARDWARE Flow Control (inbound)
            17           Use DCD Flow Control (outbound/both)
            18           Use DTR Flow Control (inbound)
            19           Use DSR Flow Control (outbound/both)
            20-127       Available for Future Use
*/
uint8_t setgetcontrol(uint8_t _ctrl)
{
  uint8_t result;

  TRACE(TRACE_INFO, F("Set/Get control %i"), (int)_ctrl);
  switch (_ctrl)
  {
  case 1:
  case 2:
  case 3:
  case 13:
  case 14:
  case 15:
  case 16:
  case 17:
  case 18:
  case 19:
    control = _ctrl;
    result = control;
  break;

  case 4:
    result = breakstate;
  break;

  case 5:
    breakstate = 1;
    result = breakstate;
  break;

  case 6:
    breakstate = 0;
    result = breakstate;
  break;

  case 7:
    result = dtrstate;
  break;

  case 8:
    dtrstate = 1;
    result = dtrstate;
  break;

  case 9:
    dtrstate = 0;
    result = dtrstate;
  break;

  case 10:
    result = rtsstate;
  break;

  case 11:
    rtsstate = 1;
    result = rtsstate;
  break;

  case 12:
    rtsstate = 0;
    result = rtsstate;
  break;

  default:
    TRACE(TRACE_ERROR, F("Unsupported control %i"), (int)_ctrl);
  case 0:
    result = control; // ziskame aktualni stav
  break;
  }

  return result;
}

// Inicializace virtualniho serioveho portu NVT
void vsp_init()
{

  Serial.begin(INITIAL_SERIAL_SPEED);
#ifdef RX_BUFFER_SIZE
  int rxsize = Serial.setRxBufferSize(RX_BUFFER_SIZE);
  TRACE(TRACE_INFO, F("RX buffser size set to %i"), rxsize);
#endif
  Serial.setDebugOutput(false);

  // Inicializace NVT parametru
  netterm.init(setserialport, sputchar, nputchar);
  netterm.setSetGetCtrl(setgetcontrol);
  server.begin();
}

void vsp_loop()
{
  int len;

  // kontrola, zda se nepripojil kilent
  if (server.hasClient())
  {
    if (!client || !client.connected())
    {
      if (client)
        client.stop();
      client = server.available();
      client.setNoDelay(true);
      ++newguy;
      TRACE(TRACE_INFO, "TCP: New client");
    }
    WiFiClient serverClient = server.available();
    serverClient.stop();
  }

  // testovani, zda ma UART nejaka data
  if (int len = Serial.available())
  {
    size_t will_copy = (len < sizeof(sbuf)) ? len : sizeof(sbuf);
    Serial.readBytes(sbuf, will_copy);
    netterm.createsendstream(sbuf, will_copy);
    serial_rx += will_copy;
  }

  // testovani, zda TCP klient neposlal nejaka data
  if (client && client.connected())
  {
    if (len = client.available())
    {
      while ((len > 0) && ((fromNet.room() >= len) || (fromNet.room() >= sizeof(sbuf))))
      {
        size_t will_copy = (len < sizeof(sbuf)) ? len : sizeof(sbuf);
        client.readBytes(sbuf, will_copy);
//        TRACE(TRACE_INFO, F("RX:%i"), will_copy);
        netterm.handlestream(sbuf, will_copy);
        len -= will_copy;
      }
    }
  }

  // testovani, zda neposlat nejaka data do TCP klienta
  if (!toNet.empty() && nvtt.expired())
  {
    size_t read = toNet.read((char *)&nvttx[0], sizeof(nvttx));

    if (client && client.connected())
    {
      uint32_t t = micros();
      client.write((const uint8_t *)&nvttx[0], read); // data do TCP soketu
//      TRACE(TRACE_INFO, F("TX: %i, took %i us"), read, (int)micros() - t);
    }
  }

  // testovani, zda neposlat nejaka data do serioveho rozhrani
  if (!fromNet.empty() && (len = Serial.availableForWrite()))
  {
  	  size_t will_copy = (len < sizeof(sbuf)) ? len : sizeof(sbuf);
  	  will_copy = fromNet.read((char *)sbuf, will_copy);
  	  Serial.write(sbuf, will_copy);
  	  serial_tx += will_copy;
  }

  if (newguy)
  {
  	newguy = 0;
  	netterm.newguy();
  }
}

