/**
 * Layout of crash data saved to EEPROM (flash)
 *
 * 1. Crash counter / how many crashes are saved
 * 2. Next avialable space in EEPROM to write data
 * 3. Crash Data Set 1
 * 4. Crash Data Set 2
 * 5. ...
 */
#define SAVE_CRASH_COUNTER          0x00  // 1 byte
#define SAVE_CRASH_WRITE_FROM       0x01  // 2 bytes
#define SAVE_CRASH_DATA_SETS        0x03  // begining of crash data sets
// Crash Data Set 1                       // variable length
// Crash Data Set 2                       // variable length
// ...                                    // variable length

/**
 * Structure of the single crash data set
 *
 *  1. Crash time
 *  2. Restart reason
 *  3. Exception cause
 *  4. epc1
 *  5. epc2
 *  6. epc3
 *  7. excvaddr
 *  8. depc
 *  9. adress of stack start
 * 10. adress of stack end
 * 11. stack trace bytes
 *     ...
 */
#define SAVE_CRASH_CRASH_TIME       0x00  // 4 bytes
#define SAVE_CRASH_RESTART_REASON   0x04  // 1 byte
#define SAVE_CRASH_EXCEPTION_CAUSE  0x05  // 1 byte
#define SAVE_CRASH_EPC1             0x06  // 4 bytes
#define SAVE_CRASH_EPC2             0x0A  // 4 bytes
#define SAVE_CRASH_EPC3             0x0E  // 4 bytes
#define SAVE_CRASH_EXCVADDR         0x12  // 4 bytes
#define SAVE_CRASH_DEPC             0x16  // 4 bytes
#define SAVE_CRASH_STACK_START      0x1A  // 4 bytes
#define SAVE_CRASH_STACK_END        0x1E  // 4 bytes
#define SAVE_CRASH_STACK_TRACE      0x22  // variable

extern "C" void custom_crash_callback(struct rst_info * rst_info, uint32_t stack, uint32_t stack_end)
{
  uint8_t crashCounter = EEPROM.read(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER);
  int16_t writeFrom;
  if (0 == crashCounter)
  {
    writeFrom = SAVE_CRASH_DATA_SETS;
  }
  else
  {
    EEPROM.get(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_WRITE_FROM, writeFrom);
  }

  // is there free EEPROM space avialable to save data for this crash?
  if (writeFrom + SAVE_CRASH_STACK_TRACE > SAVE_CRASH_SPACE_SIZE)
  {
    return;
  }

  // increment crash counter and write it to EEPROM
  EEPROM.write(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER, ++crashCounter);

  // now address EEPROM contents including SAVE_CRASH_EEPROM_OFFSET
  writeFrom += SAVE_CRASH_EEPROM_OFFSET;

  // write crash time to EEPROM
  uint32_t crashTime = millis();
  EEPROM.put(writeFrom + SAVE_CRASH_CRASH_TIME, crashTime);

  // write reset info to EEPROM
  EEPROM.write(writeFrom + SAVE_CRASH_RESTART_REASON, rst_info->reason);
  EEPROM.write(writeFrom + SAVE_CRASH_EXCEPTION_CAUSE, rst_info->exccause);

  // write epc1, epc2, epc3, excvaddr and depc to EEPROM
  EEPROM.put(writeFrom + SAVE_CRASH_EPC1, rst_info->epc1);
  EEPROM.put(writeFrom + SAVE_CRASH_EPC2, rst_info->epc2);
  EEPROM.put(writeFrom + SAVE_CRASH_EPC3, rst_info->epc3);
  EEPROM.put(writeFrom + SAVE_CRASH_EXCVADDR, rst_info->excvaddr);
  EEPROM.put(writeFrom + SAVE_CRASH_DEPC, rst_info->depc);

  // write stack start and end address to EEPROM
  EEPROM.put(writeFrom + SAVE_CRASH_STACK_START, stack);
  EEPROM.put(writeFrom + SAVE_CRASH_STACK_END, stack_end);

  // write stack trace to EEPROM
  int16_t currentAddress = writeFrom + SAVE_CRASH_STACK_TRACE;
  for (uint32_t iAddress = stack; iAddress < stack_end; iAddress++)
  {
    uint8_t* byteValue = (uint8_t*) iAddress;
    EEPROM.write(currentAddress++, *byteValue);
    if (currentAddress - SAVE_CRASH_EEPROM_OFFSET > SAVE_CRASH_SPACE_SIZE)
    {
      // ToDo: flag an incomplete stack trace written to EEPROM!
      break;
    }
  }
  // now exclude SAVE_CRASH_EEPROM_OFFSET from address written to EEPROM
  currentAddress -= SAVE_CRASH_EEPROM_OFFSET;
  EEPROM.put(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_WRITE_FROM, currentAddress);

  EEPROM.commit();
  return;
}

/**
 * Clear crash information saved in EEPROM
 * In fact only crash counter is cleared
 * The crash data are not deleted
 */
void crashClear(void)
{

  // clear the crash counter
  EEPROM.write(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER, 0);
  EEPROM.commit();
}

/**
 * Print out crash information that has been previusly saved in EEPROM
 */
void crashGet(String &out)
{
  uint8_t crashCounter = EEPROM.read(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER);

  if (crashCounter == 0)
  {
    out.concat(F("No crashes saved"));
    return;
  }

  char buff[256];

  out.concat(F("Crash information recovered from EEPROM<br>"));
  int16_t readFrom = SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_DATA_SETS;
  for (uint8_t k = 0; k < crashCounter; k++)
  {
    uint32_t crashTime;
    EEPROM.get(readFrom + SAVE_CRASH_CRASH_TIME, crashTime);
    sprintf_P(buff, PSTR("Crash # %d at %ld ms<br>"), k + 1, crashTime);
    out.concat(buff);

    out.concat(F("Reason of restart: "));
    out.concat(EEPROM.read(readFrom + SAVE_CRASH_RESTART_REASON));
    out.concat(F("<br>Exception cause: "));
    out.concat(EEPROM.read(readFrom + SAVE_CRASH_EXCEPTION_CAUSE));
    out.concat(F("<br>"));

    uint32_t epc1, epc2, epc3, excvaddr, depc;
    EEPROM.get(readFrom + SAVE_CRASH_EPC1, epc1);
    EEPROM.get(readFrom + SAVE_CRASH_EPC2, epc2);
    EEPROM.get(readFrom + SAVE_CRASH_EPC3, epc3);
    EEPROM.get(readFrom + SAVE_CRASH_EXCVADDR, excvaddr);
    EEPROM.get(readFrom + SAVE_CRASH_DEPC, depc);
    sprintf_P(buff, PSTR("epc1=0x%08x epc2=0x%08x epc3=0x%08x excvaddr=0x%08x depc=0x%08x<br>"), epc1, epc2, epc3, excvaddr, depc);
    out.concat(buff);

    uint32_t stackStart, stackEnd;
    EEPROM.get(readFrom + SAVE_CRASH_STACK_START, stackStart);
    EEPROM.get(readFrom + SAVE_CRASH_STACK_END, stackEnd);
    out.concat(F("&gt;&gt;&gt;stack&gt;&gt;&gt;<br>"));
    int16_t currentAddress = readFrom + SAVE_CRASH_STACK_TRACE;
    int16_t stackLength = stackEnd - stackStart;
    uint32_t stackTrace;
    for (int16_t i = 0; i < stackLength; i += 0x10)
    {
      sprintf_P(buff, PSTR("%08x: "), stackStart + i);
      out.concat(buff);
      for (uint8_t j = 0; j < 4; j++)
      {
        EEPROM.get(currentAddress, stackTrace);
        sprintf_P(buff, PSTR("%08x "), stackTrace);
        out.concat(buff);
        currentAddress += 4;
        if (currentAddress - SAVE_CRASH_EEPROM_OFFSET > SAVE_CRASH_SPACE_SIZE)
        {
          out.concat(F("<br>Incomplete stack trace saved!"));
          goto eepromSpaceEnd;
        }
      }
      out.concat(F("<br>"));
    }
    eepromSpaceEnd:
    out.concat(F("&lt;&lt;&lt;stack&lt;&lt;&lt;<br>"));
    readFrom = readFrom + SAVE_CRASH_STACK_TRACE + stackLength;
  }
  int16_t writeFrom;
  EEPROM.get(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_WRITE_FROM, writeFrom);
//  EEPROM.end();

  // is there free EEPROM space avialable to save data for next crash?
  if (writeFrom + SAVE_CRASH_STACK_TRACE > SAVE_CRASH_SPACE_SIZE)
  {
    out.concat(F("No more EEPROM space available to save crash information!"));
  }
  else
  {
    sprintf_P(buff, PSTR("EEPROM space available: 0x%04x bytes<br>"), SAVE_CRASH_SPACE_SIZE - writeFrom);
    out.concat(buff);
  }
}

int crashGetCount(void)
{
  uint8_t crashCounter = EEPROM.read(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER);

  return (int)crashCounter;
}

int crashSanityCheck(void)
{
  int res = 0; // predpokladame, ze je vysledek ok

  uint8_t crashCounter = EEPROM.read(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_COUNTER);
  int16_t writeFrom;
  EEPROM.get(SAVE_CRASH_EEPROM_OFFSET + SAVE_CRASH_WRITE_FROM, writeFrom);

  if ((crashCounter > 32) && (writeFrom >= SAVE_CRASH_SPACE_SIZE))
    res = 1; // patrne spatne inicializovana oblast
  return res;
}
