// Kompilace verzi 1.6.9

#include "configuration.h"
#include <EEPROM.h>
#include "src/TimerOne/TimerOne.h"
#include "src/MultiFuncShield/MultiFuncShield.h"
#include "src/RS485/RS485.h"
#include "src/interval/interval.h"

template <class T> int EEPROM_writeAnything(int ee, const T& value)
{
   const uint8_t* p = (const uint8_t*)(const void*)&value;
   int i;
   for (i = 0; i < sizeof(value); i++)
       EEPROM.write(ee++, *p++);
   return i;
}

template <class T> int EEPROM_readAnything(int ee, T& value)
{
   uint8_t* p = (uint8_t*)(void*)&value;
   int i;
   for (i = 0; i < sizeof(value); i++)
       *p++ = EEPROM.read(ee++);
   return i;
}

// Definice obsazeni EEPROM
#define EEPROM_CONFIG_ORIGIN 0 // pocatek konfigurace od zacatku EEPROM

typedef struct
{
  uint32_t zeropos; // nulova hladina (vycitana z EEPROM, nastavovana v setup menu)
  int level; // kalibracni hladina
  uint32_t levelpos; // hodnota z A/D prevodniku, ktera odpovida hodnote level
  int level_1; // voda nedostatek
  int level_2; // voda dostatek
  int level_3; // voda nadbytek
} eepromdata_t;

enum netResults
{
  NET_NOTHING = 0, // zadna zprava neni prijata
  NET_MESSAGE, // prijata zprava
  NET_TIMEOUT, // doslo k timeoutu cekani na zpravu
};

enum displayModeValues
{
  MODE_DISPLAY_LEVEL, // zobrazeni vysky hladiny
  MODE_DISPLAY_TEMPERATURE, // zobrazeni teploty
  MODE_DISPLAY_ERROR, // priznak zobrazeni chyby (problem se senzorem)
  MODE_DISPLAY_SETUP, // zobrazeni nastavovaciho menu
  MODE_IN_SETUP, // modifikace Setup polozky
  MODE_IN_SET_ZERO, // modifikace nulove hladiny (vahy merici tyce)
};

uint8_t displayMode = MODE_DISPLAY_LEVEL;

enum setupMode
{
	SETUP_LEVEL1 = 0, // nizka hladina vody [cm]
	SETUP_LEVEL2, // jakasyk jina hladina [cm]
  SETUP_LEVEL3, // jakasyk jina hladina [cm]
	SETUP_ZERO_LEVEL, // nulova hladina (volne visici sonda)
	SETUP_LEVEL, // nastaveni aktualni hladiny [cm]
	SETUP_RE1, // test rele 1
	SETUP_RE2, // test rele 2
	SETUP_EXIT // ukonceni nastaveni
};

uint8_t menuMode = SETUP_LEVEL1;

enum networkModeValues
{
  NM_IDLE, // klidovy stav
  NM_WAITING,
};

uint8_t networkMode = NM_IDLE;

uint32_t lastSendTime; // casovac pro casovani komunikace

Interval meas; // casovaci interval pro mereni

long adcval; // posledni znama hodnota z A/D prevodniku
int adclevel; // aktualni hladina v jednotkach (asi cm)
int temp; // posledni znama teplota senzoru (celociselna cast)
float f; // pomocna promenna pro vypocet

// ---- Ulozeno v EEPROM
uint32_t zeropos; // nulova hladina (vycitana z EEPROM, nastavovana v setup menu)
int level; // kalibracni hladina
uint32_t levelpos; // odpovidajici hodnota z A/D prevodniku, ktera odpovida hodnote level
int level_1;
int level_2;
int level_3;
// ---- Konec ulozeni v EEPROM
int value; // hodnota, se kterou pracuje setup pri zvysovani/snizovani hodnoty
int fastinc = 0; // rychly posun pri drzeni tlacitka
Interval fr; // casovac pro opakovani (rychla zmena hodnoty)

uint8_t re1status = 1;
uint8_t re2status = 1;

// zpozdeni pro dokonceni otaceni linky RS485 z protistrany
void pause(void)
{

  delay(50);
}

void netSend(char *s)
{

  if (NM_IDLE == networkMode)
  {
  	if ((millis() - lastSendTime) < 30)
  	  delay(30); // cekame na dokonceni posledniho prenosu
  	RS485_Send(s); // odesleme zpravu
  	meas.set(SENSOR_MEASURE_TIMEOUT); // nastavime timeout
  	networkMode = NM_WAITING;
  	lastSendTime = millis();
  }
}

uint8_t netReceive(uint8_t **ptr)
{

  if (NM_WAITING == networkMode)
  {
  	if (meas.expired())
  	{
  	  networkMode = NM_IDLE; // timeout odpovedi
  	  return NET_TIMEOUT; // chyba prenosu
  	}
  	uint8_t *rm = RS485_Receive(); // zkusime prijmout zpravu
  	if (rm)
  	{
  	  *ptr = rm; // ulozime ukazatel prijate zpravy
  	  networkMode = NM_IDLE;
  	  lastSendTime = millis(); // automaticke hlidani pro otaceni RS485 budice
  	  return NET_MESSAGE; // prijata zprava
  	}
  }
  return NET_NOTHING; // nic neni prijate
}

void displayLevel(void)
{

  MFS.blinkDisplay(DIGIT_ALL, OFF);
  if (adclevel < 0)
  {
    MFS.write("0.0",1);
  }
  else
  {
    char disp[8];
    int dp, lev;

    lev = adclevel / 100;
    dp = adclevel % 100;
    sprintf(disp, "%d.%02d", lev, dp);
    MFS.write(disp, 1);
  }
}

void displayTemperature(void)
{

  MFS.blinkDisplay(DIGIT_ALL, OFF);
  String s = String(temp + TEMP_OFFSET);
  s.concat("&");
  MFS.write(s.c_str(), 1);
}

void displaySetup(void)
{

  MFS.writeLeds(LED_ALL, OFF);
  MFS.blinkLeds(LED_ALL, ON);
	switch (menuMode)
	{
	case SETUP_LEVEL1:
    MFS.write("Le 1", 1);
    MFS.writeLeds(LED_4, ON);
	break;

	case SETUP_LEVEL2:
    MFS.write("Le 2", 1);
    MFS.writeLeds(LED_3, ON);
	break;

  case SETUP_LEVEL3:
    MFS.write("Le 3", 1);
    MFS.writeLeds(LED_1, ON);
  break;

	case SETUP_ZERO_LEVEL:
    MFS.write("Nula", 1);
	break;

	case SETUP_LEVEL:
    MFS.write("Hlad", 1);
	break;

	case SETUP_RE1:
	{
	  String s = "RE ";

	  if (!re1status)
		s.concat('.');
	  s.concat('1');
    MFS.write(s.c_str(), 1);
	}
	break;

	case SETUP_RE2:
	{
      String s = "RE ";

      if (!re2status)
        s.concat('.');
	  s.concat('2');
      MFS.write(s.c_str(), 1);
	}
	break;

	case SETUP_EXIT:
    MFS.write("End", 1);
	break;

	default:
    MFS.write("???", 1);
	break;
	}
}

void displayValue(void)
{

  MFS.write(value);
}

//--------------------------------------------------------------------------------------------------
void setup()
{

  Timer1.initialize();
  MFS.initialize(&Timer1);

  pinMode(RE1, OUTPUT);
  digitalWrite(RE1, re1status);
  pinMode(RE2, OUTPUT);
  digitalWrite(RE2, re2status);

  MFS.beep(50);
  MFS.write(INOVER, 1);
  delay(8000);

  RS485_Begin(19200); // aktivujeme RS485

  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, zeropos), zeropos);
  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level), level);
  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, levelpos), levelpos);
  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_1), level_1);
  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_2), level_2);
  EEPROM_readAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_3), level_3);

  netSend("m"); // ziskame mereni ze senzoru
  meas.set(SENSOR_MEASURE_TIMEOUT); // timeout pro mereni
}

void loop()
{

  uint8_t btn = MFS.getButton(); // stav tlacitek
  uint8_t received = 0; // prijem zpravy ze site (ze senzoru) - signalizuje ziskani nove hodnoty mereni
  uint8_t *ptr;
  uint8_t result = netReceive(&ptr);

  switch (result)
  {
  	case NET_MESSAGE:
      switch (ptr[0])
      {
      case 'M':
      {
      	int dummy;

        sscanf((char *)ptr, "M:%ld,%ld.%ld", &adcval, &temp, &dummy);
  		  long actual = zeropos - adcval;
  		  f = (float)level / (zeropos - levelpos);
  		  adclevel = f * actual;

        received = 1; // mame nova data z mereni

        uint8_t leds = 0;
        if (adclevel < level_1)
        	leds = LED_4; // nedostatek vody
        else if (adclevel < level_2)
        	leds = LED_3; // dostatek vody
        else if (adclevel >= level_3)
          leds = LED_1; // nadbytek vody
        else
        	leds = LED_2; // normal vody
        if ((MODE_DISPLAY_LEVEL == displayMode) || (MODE_DISPLAY_TEMPERATURE == displayMode))
        {
          MFS.writeLeds(LED_ALL, OFF);
          MFS.writeLeds(leds, ON);
        }
      }
      break;

      default:
      break;
      }
      netSend("m");
  	break;

  	case NET_TIMEOUT:
      netSend("m");
      if ((MODE_DISPLAY_TEMPERATURE == displayMode) || (MODE_DISPLAY_LEVEL == displayMode))
      {
      	displayMode = MODE_DISPLAY_ERROR; // detekovali jsme chybu
      	MFS.blinkDisplay(DIGIT_ALL);
      	MFS.write("Err", 1);
  	  }
  	break;
  }

  switch (displayMode)
  {
  case MODE_DISPLAY_ERROR:
    if (received)
    { // chyba zmizela
    	displayMode = MODE_DISPLAY_LEVEL; // prechazime do normalniho rezimu
    	displayLevel(); // zobrazime hladinu
    }
    if (btn == BUTTON_2_LONG_PRESSED)
    {
      menuMode = SETUP_LEVEL1;
      displayMode = MODE_DISPLAY_SETUP;
  	  MFS.blinkDisplay(DIGIT_ALL, OFF);
      displaySetup(); // zobrazime prvni polozku menu
    }
  break;

  case MODE_DISPLAY_LEVEL:
    if (received)
    {
    	displayLevel(); // prisla nova hladina - zobrazime ji
    }
    if (btn == BUTTON_2_LONG_PRESSED)
    { // prechazime do systemu menu
      menuMode = SETUP_LEVEL1;
      displayMode = MODE_DISPLAY_SETUP;
      displaySetup(); // zobrazime prvni polozku menu
    }
    if ((btn == BUTTON_1_PRESSED) || (btn == BUTTON_2_PRESSED))
    {
    	displayMode = MODE_DISPLAY_TEMPERATURE;
    	displayTemperature();
    }
  break;

  case MODE_DISPLAY_TEMPERATURE:
    if (received)
    {
    	displayTemperature();
    }
    if (btn == BUTTON_2_LONG_PRESSED)
    { // prechazime do systemu menu
      menuMode = SETUP_LEVEL1;
      displayMode = MODE_DISPLAY_SETUP;
      displaySetup(); // zobrazime prvni polozku menu
    }
    if ((btn == BUTTON_1_PRESSED) || (btn == BUTTON_2_PRESSED))
    {
    	displayMode = MODE_DISPLAY_LEVEL;
    	displayLevel();
    }
  break;

  case MODE_DISPLAY_SETUP:
    if (btn == BUTTON_1_PRESSED)
    {
    	menuMode++;
    	if (menuMode > SETUP_EXIT)
    		menuMode = 0;
    	displaySetup();
    }
    if (btn == BUTTON_2_PRESSED)
    {
    	menuMode--;
    	if (menuMode > SETUP_EXIT)
    		menuMode = SETUP_EXIT;
    	displaySetup();
    }
    if (btn == BUTTON_3_SHORT_RELEASE)
    {
    	switch (menuMode)
    	{
    	case SETUP_EXIT:
    		displayMode = MODE_DISPLAY_LEVEL;
        MFS.blinkLeds(LED_ALL, OFF);
    		displayLevel();
    	break;

    	case SETUP_ZERO_LEVEL:
        displayMode = MODE_IN_SET_ZERO;
        MFS.write("Set", 1); // zobrazime text
    	break;

    	case SETUP_LEVEL:
    	  value = level;
    	  fastinc = 0;
    	  displayMode = MODE_IN_SETUP;
    	  displayValue();
    	break;

    	case SETUP_LEVEL1: // nedostatek vody (LED_4)
    	  value = level_1;
    	  fastinc = 0;
    	  displayMode = MODE_IN_SETUP;
    	  displayValue();
    	break;

    	case SETUP_LEVEL2: // dostatek vody (LED_2)
    	  value = level_2;
    	  fastinc = 0;
    	  displayMode = MODE_IN_SETUP;
    	  displayValue();
    	break;

      case SETUP_LEVEL3: // nadbytek vody (LED_1)
    	  value = level_3;
    	  fastinc = 0;
    	  displayMode = MODE_IN_SETUP;
    	  displayValue();
    	break;

    	case SETUP_RE1:
    		re1status = !re1status;
    		digitalWrite(RE1, re1status);
    		displaySetup();
    	break;

    	case SETUP_RE2:
    		re2status = !re2status;
    		digitalWrite(RE2, re2status);
    		displaySetup();
    	break;

    	default:
    	break;
    	}
    }
  break;

  case MODE_IN_SETUP:
    if (btn == BUTTON_1_PRESSED)
    {
    	++value;
    	if (value > 999)
    		value = 0;
    	displayValue();
    }
    if (btn == BUTTON_2_PRESSED)
    {
    	--value;
    	if (value < 0)
    		value = 999;
    	displayValue();
    }
    if (btn == BUTTON_1_LONG_PRESSED)
    	fastinc = 10;
    if (btn == BUTTON_2_LONG_PRESSED)
    	fastinc = -10;
    if ((btn == BUTTON_1_LONG_RELEASE) || (btn == BUTTON_2_LONG_RELEASE))
    	fastinc = 0;
    if (btn == BUTTON_3_LONG_PRESSED)
    { // zruseni editace
    	displayMode = MODE_DISPLAY_SETUP;
    	displaySetup();
    	fastinc = 0;
    }
    if (btn == BUTTON_3_SHORT_RELEASE)
    { // ulozeni hodnoty a prechod do vyssi urovne nastaveni
    	switch (menuMode)
    	{
    	case SETUP_LEVEL:
    	  level = value; // nastavime si hodnotu
    	  levelpos = adcval;
  			EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level), level);
  			EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, levelpos), levelpos);
    	break;

    	case SETUP_LEVEL1:
    	  level_1 = value; // nastavime si hodnotu
  			EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_1), level_1);
    	break;

    	case SETUP_LEVEL2:
    	  level_2 = value; // nastavime si hodnotu
  			EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_2), level_2);
    	break;

    	case SETUP_LEVEL3:
    	  level_3 = value; // nastavime si hodnotu
  			EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, level_3), level_3);
    	break;

    	}
    	displayMode = MODE_DISPLAY_SETUP;
    	displaySetup();
    }
    if (fastinc)
    	if (fr.expired())
    	{
    		value += fastinc;
    		if (value > 999)
    		{
    			value = 999;
    			fastinc = 0;
    		}
    		if (value < 0)
    		{
    			value = 0;
    			fastinc = 0;
    		}
    		displayValue();
    		fr.set(100);
    	}
  break;

  case MODE_IN_SET_ZERO:
    switch (btn)
    {
    case BUTTON_1_LONG_PRESSED: // pouze dlouze stisknute tlacitko 1 ulozi hodnotu
      MFS.beep(50);
      zeropos = adcval; // nastavime nulovou hladinu
      EEPROM_writeAnything(EEPROM_CONFIG_ORIGIN + offsetof(eepromdata_t, zeropos), zeropos);
      displayMode = MODE_DISPLAY_SETUP;
      displaySetup();
    break;

    case BUTTON_1_SHORT_RELEASE:
    case BUTTON_2_PRESSED:
    case BUTTON_3_SHORT_RELEASE:
    case BUTTON_2_LONG_PRESSED:
    case BUTTON_3_LONG_PRESSED:
      displayMode = MODE_DISPLAY_SETUP;
      displaySetup();
      MFS.beep(20, 20, 1, 3);
    break;
    }
  break;

  default:
    displayMode = MODE_DISPLAY_ERROR; // problem se stavem menu
  break;
  }
}

