/**
 * Test zobrazovani JPEG obrazku pro ESP8266
 *
 * @file jpgt.ino
 * @author Pavel Brychta, http://www.xpablo.cz
 *
 * Copyright (c) 2016 Pavel Brychta. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "configuration.h"
#include <EEPROM.h>
#include <ESP8266WiFi.h>
#include <ESP8266WebServer.h>
#include <ArduinoOTA.h>
#include <ESP8266mDNS.h>
#include <Ticker.h>
#include <SPI.h>
#ifdef HTTP_UPDATE
 #include <ESP8266HTTPUpdateServer.h>
#endif
#include "src/interval/interval.h"
#include "src/espnbns/espnbns.h"
#include "src/WiFiConfig/WiFiConfig.h"
#if (LED_PIN != NO_PIN)
 #include "src/led/led.h"
#endif
#include "embeddedHTML.h"
#ifdef USE_SPIFFS
 #warning Pouzivame SPIFFS - nezapomen pripravit obsah flash!
 #include "FS.h"
#endif
#include "src/JPEGDecoder/JPEGDecoder.h"
#include "ili9340.h"

// Definice obsazeni EEPROM
#define elementSize(type, element) sizeof(((type *)0)->element)
typedef struct
{
  wificonfigarea_t wc; // oblast, vyhrazena pro konfiguraci WiFi
// **** sem pokracuji dalsi polozky, ukladane do EEPROM

} eepromdata_t;

#if (LED_PIN != NO_PIN)
 LED led(LED_PIN, LOW, HIGH);
 const uint8_t LS_CONNECTING[] = {LEDS_ONFOR + 0, LEDS_OFFFOR + 8, LEDS_RESTART};
 const uint8_t LS_CONFIGAP[] = {LEDS_ONFOR + 0, LEDS_OFFFOR + 0, LEDS_ONFOR + 0, LEDS_OFFFOR + 6, LEDS_RESTART};
 const uint8_t LS_CONNECTED[] = {LEDS_ONFOR + 0, LEDS_OFFFOR + 0, LEDS_ONFOR + 0, LEDS_OFFFOR + 0, LEDS_ONFOR + 0, LEDS_OFFFOR + 6, LEDS_RESTART};
#endif

#ifdef HTTP_UPDATE
 ESP8266HTTPUpdateServer httpUpdater;
#endif

int otaActive = 0; // priznak moznosti aktivovat OTA
NBNS nbns; // Netbios
ESP8266WebServer www(HTTP_PORT); // webovy server
// **** sem je mozne dopsat dalsi globalni promenne

inline uint16_t RGB565(uint8_t r, uint8_t g, uint8_t b)
{

//  return ((r / 8) << 11) | ((g / 4) << 5) | (b / 8);
// blue >> 0 | ((green << 5) & 0x7E0) | ((red << 11) & 0xF800
  return ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
}

// http://blog.boochow.com/article/427690966.html
// http://www.toughdev.com/content/2014/03/using-picojpeg-library-on-a-pic-with-ili9341-320x240-lcd-module/
void jpegDraw(char* filename)
{
  char str[100];
  uint8 *pImg;
  int x,y,bx,by;

  // Decoding start
  TRACE(TRACE_INFO, F("Decode: %i"), JpegDec.decode(filename,0));

  // Image Information
  TRACE(TRACE_INFO, F("Width %i"), (int)JpegDec.width);
  TRACE(TRACE_INFO, F("Height %i"), (int)JpegDec.height);
  TRACE(TRACE_INFO, F("Components %i"), (int)JpegDec.comps);
  TRACE(TRACE_INFO, F("MCU / row %i"), (int)JpegDec.MCUSPerRow);
  TRACE(TRACE_INFO, F("MCU / col %i"), (int)JpegDec.MCUSPerCol);
  TRACE(TRACE_INFO, F("Scan type %i"), (int)JpegDec.scanType);
  TRACE(TRACE_INFO, F("MCU width %i"), (int)JpegDec.MCUWidth);
  TRACE(TRACE_INFO, F("MCU height %i"), (int)JpegDec.MCUHeight);

  // Raw Image Data
  while(JpegDec.read()){
    pImg = JpegDec.pImage ;

    for(by=0; by<JpegDec.MCUHeight; by++)
    {
      for(bx=0; bx<JpegDec.MCUWidth; bx++)
      {

        x = JpegDec.MCUx * JpegDec.MCUWidth + bx;
        y = JpegDec.MCUy * JpegDec.MCUHeight + by;

        if(x<JpegDec.width && y<JpegDec.height){
          if(JpegDec.comps == 1){ // Grayscale
            ili9340_drawPixel(x, y, RGB565(pImg[0], pImg[0], pImg[0]));
          }else{ // RGB
            ili9340_drawPixel(x, y, RGB565(pImg[0], pImg[1], pImg[2]));
          }
        }
        pImg += JpegDec.comps ;
        wdt_reset();
      }
    }
  }
}

void wificfgcb(wificonfigstate_t state)
{

  switch (state)
  {
    case WCS_CONNECTSTART:
    // **** kod pro start signalizace, oznamujici zacatek pripojovani k WiFi siti (volano pouze jednou)
#if (LED_PIN != NO_PIN)
      led.set(LS_CONNECTING);
#endif

      break;

    case WCS_CONNECTING:
    // **** kod pro periodickou signalizaci probihajiciho pripojovani k WiFi siti (volano periodicky)

      break;

    case WCS_CONNECTED:
    // **** kod pro start signalizace uspesneho pripojeni k WiFi siti (volano pouze jednou)
#if (LED_PIN != NO_PIN)
      led.set(LS_CONNECTED);
#endif

      break;

    case WCS_CONFIGSTART:
    // **** kod pro start signalizace, oznamujici spusteni konfiguracniho AP (volano pouze jednou)
#if (LED_PIN != NO_PIN)
      led.set(LS_CONFIGAP);
#endif

      break;

    case WCS_CONFIGWAIT:
    // **** kod pro periodickou signalizaci beziciho konfiguracniho AP (volano periodicky)

      break;
  }
}

void ICACHE_FLASH_ATTR setup()
{
  int _fc;

  TRACE_INIT; // inicializace ladeni
  EEPROM.begin(sizeof(eepromdata_t) + 10); // zahajujeme praci s EEPROM (10 bytu je jen rezerva)
#if (LED_PIN != NO_PIN)
  led.begin(); // inicializace signalizace
#endif

#if (FORCE_CONFIG_BUTTON_PIN != NO_PIN)
  pinMode(FORCE_CONFIG_BUTTON_PIN, INPUT_PULLUP);
  delay(20); // male zpozdeni, aby se ustalila hodnota na vstupu
  _fc = digitalRead(FORCE_CONFIG_BUTTON_PIN); // pokud je na I/O pinu hodnota 0, tak vynutime nastavovaci AP
#else
  _fc = 1; // nevstupujeme do konfigurace
#endif
  {
    WiFiConfig wifi; // konfigurace WiFi casti ESP modulu

    if (WCR_OK != wifi.begin(offsetof(eepromdata_t, wc), _fc, 60, wificfgcb)) // startujeme pripojeni
      ESP.restart();
  }
  if (ESP.getFlashChipRealSize() > 1000000)
    otaActive = 1; // flash pameti je dost - povolime OTA

  if (strlen(WiFiDeviceName))
  {
    nbns.begin(WiFiDeviceName);
    if (otaActive)
    {
      ArduinoOTA.setHostname(WiFiDeviceName);
      ArduinoOTA.begin();
      TRACE(TRACE_INFO, F("OTA aktivovano"));
    }
    else
    {
  	  MDNS.begin(WiFiDeviceName);
      MDNS.addService("http", "tcp", HTTP_PORT);
    }
  }
// montaz souboroveho systemu
#ifdef USE_SPIFFS
  if (!SPIFFS.begin())
  	TRACE(TRACE_ERROR, F("SPIFFS neni pripojeny!"));
#endif
// Start weboveho serveru - sem je mozno pridavat odkazy na dalsi stranky

  www.on("/favicon.ico", []() {
    www.send(200, FPSTR(HTML), "");
  });

  www.onNotFound([]() {
#ifdef USE_SPIFFS
    if (!handleFileRead(www.uri()))
#endif
      handleNotFound();
  });

#ifdef HTTP_UPDATE
  httpUpdater.setup(&www);
#endif

  TRACE_ADDWEB(&www); // ladici stranka
  www.begin(); // startujeme webovy server
// **** dalsi uzivatelska inicializace
  ili9340_init();
//  ili9340_setRotation(0);
  ili9340_setRotation(1);

  ili9340_drawRect(186,6,10,10,ILI9340_RED,ILI9340_BLACK);
  ili9340_drawRect(200,6,10,10,ILI9340_RED,ILI9340_BLACK);
  ili9340_drawRect(214,6,10,10,ILI9340_RED,ILI9340_BLACK);
  ili9340_drawRect(228,6,10,10,ILI9340_RED,ILI9340_BLACK);
  // delimit fixed areas
  ili9340_drawFastHLine(0,20, 240, ILI9340_BLUE);
  ili9340_drawFastHLine(0,300, 240, ILI9340_RED);

  wdt_reset();
//  jpegDraw("/test.jpg");
  jpegDraw("/vv.jpg");

}

void loop()
{

  if (otaActive)
  { // pripadna obsluha OTA aktualizace FW
    ArduinoOTA.handle();
  }
  www.handleClient();
  nbns.poll();
  TRACE_POLL;
// **** dalsi uzivatelske metody

}


